using System;
using System.Collections;
using System.Data;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using Method = System.Reflection.MethodBase;
using EQUIPMENT = gov.va.med.vbecs.Common.VbecsTables.Equipment;
using EQUIPMENTMAINTENANCE = gov.va.med.vbecs.Common.VbecsTables.EquipmentMaintenance;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for EquipmentMaintenance.
	/// </summary>
	public class EquipmentMaintenance: BaseBusinessObject
	{
		private	const string MIN_COLUMNS = EQUIPMENTMAINTENANCE.EquipmentMaintenanceGuid + "," + EQUIPMENTMAINTENANCE.EquipmentGuid + "," + EQUIPMENTMAINTENANCE.MaintenanceDate + "," + 
			EQUIPMENTMAINTENANCE.MaintenanceResults + "," + EQUIPMENTMAINTENANCE.MaintenanceType + "," + EQUIPMENTMAINTENANCE.PendingIssues + "," + EQUIPMENTMAINTENANCE.RowVersion;

		private DataRow _equipmentMaintenanceRow = null;
		private DataTable _equipmentMaintenanceTable = null;

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5668"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Valid EquipmentMaintenance object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5669"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>Argument Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// EquipmentMaintenance object constructor - Populates data from guid
		/// </summary>
		/// <param name="equipmentMaintenanceGuid">Equipment maintenance unique identfier</param>
		public EquipmentMaintenance(System.Guid equipmentMaintenanceGuid)
		{
			if (equipmentMaintenanceGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("equipment maintenance guid").ResString);
			}

			//Get the datarow for the requested guid
			this._equipmentMaintenanceTable = DAL.EquipmentMaintenance.GetEquipmentMaintenanceByGuid(equipmentMaintenanceGuid);

			if (this._equipmentMaintenanceTable.Rows.Count == 0)
			{
				this._equipmentMaintenanceRow = this._equipmentMaintenanceTable.NewRow();
				this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.EquipmentMaintenanceGuid] = equipmentMaintenanceGuid;
				this.IsNew = true;
				this.IsDirty = true;
			}
			else
			{
				this.LoadFromDataRow(this._equipmentMaintenanceTable.Rows[0]);
				this.IsNew = false;
				this.IsDirty = false;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5670"> 
		///		<ExpectedInput>Valid equipment maintenance data row</ExpectedInput>
		///		<ExpectedOutput>Valid EquipmentMaintenance object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5671"> 
		///		<ExpectedInput>Invalid data row</ExpectedInput>
		///		<ExpectedOutput>ArgumnentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// EquipmentMaintenance object constructor - Populates data from data row
		/// </summary>
		/// <param name="dr">Equipment maintenance data row</param>
		public EquipmentMaintenance(DataRow dr)
		{
			if (!Common.Utility.VerifyMinColumnsInDataTable(MIN_COLUMNS.Split(','), dr.Table))
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InsufficientColumns(EQUIPMENT.TableName).ResString);

			this._equipmentMaintenanceTable = dr.Table.Clone();
			this.LoadFromDataRow(dr);
			this.IsDirty = false;
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5672"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Empty equipment maintenance data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5673"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates a blank equipment maintenance data table
		/// </summary>
		/// <returns>Empty equipment maintenance data table</returns>
		public static System.Data.DataTable GetEmptyEquipmentMaintenanceTable()
		{
			return DAL.EquipmentMaintenance.GetEmptyEquipmentMaintenanceTable();
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5674"> 
		///		<ExpectedInput>Valid EquipmentMaintenance object</ExpectedInput>
		///		<ExpectedOutput>EquipmentMaintenanceGuid property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5675"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets Equipment Maintenance Guid (Read-Only, should be set during MakeNew() of class)
		/// </summary>
		public Guid EquipmentMaintenceGuid
		{
			get
			{
				if (this._equipmentMaintenanceRow.IsNull(EQUIPMENTMAINTENANCE.EquipmentMaintenanceGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.EquipmentMaintenanceGuid];
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5676"> 
		///		<ExpectedInput>Valid EquipmentMaintenance object</ExpectedInput>
		///		<ExpectedOutput>EquipmentGuid property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5677"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Equipment Guid
		/// </summary>
		public Guid EquipmentGuid
		{
			get
			{
				if (this._equipmentMaintenanceRow.IsNull(EQUIPMENTMAINTENANCE.EquipmentGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.EquipmentGuid];
			}
			set
			{
				this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.EquipmentGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5678"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>MaintenanceActivity property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5679"> 
		///		<ExpectedInput>Invalid string (51 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Maintenance Activity Type
		/// </summary>
		public string MaintenanceActivity
		{
			get
			{
				if (this._equipmentMaintenanceRow.IsNull(EQUIPMENTMAINTENANCE.MaintenanceType))
					return string.Empty;

				return this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.MaintenanceType].ToString();
			}
			set
			{
				if (value.Length > 50)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "50").ResString;
					throw new ArgumentException(errorMessage);
				}

				this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.MaintenanceType] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 50)));
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5680"> 
		///		<ExpectedInput>Valid date</ExpectedInput>
		///		<ExpectedOutput>MaintenanceDate property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5681"> 
		///		<ExpectedInput>Invalid date/time (future date/time)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Maintenance Date/Time
		/// </summary>
		public System.DateTime MaintenanceDate
		{
			get
			{
				if (this._equipmentMaintenanceRow.IsNull(EQUIPMENTMAINTENANCE.MaintenanceDate))
					return System.DateTime.MinValue;

				return (System.DateTime) this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.MaintenanceDate];
			}
			set
			{
				if (value > BOL.VBECSDateTime.GetDivisionCurrentDateTime())
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(MethodBase.GetCurrentMethod().Name).ResString);

				this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.MaintenanceDate] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value == System.DateTime.MinValue)||(value>BOL.VBECSDateTime.GetDivisionCurrentDateTime())));
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5682"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>MaintenanceResult property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5683"> 
		///		<ExpectedInput>Invalid string (256 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Maintenance Result
		/// </summary>
		public string MaintenanceResult
		{
			get
			{
				if (this._equipmentMaintenanceRow.IsNull(EQUIPMENTMAINTENANCE.MaintenanceResults))
					return string.Empty;

				return this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.MaintenanceResults].ToString();
			}
			set
			{
				if (value.Length > 255)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "255").ResString;
					throw new ArgumentException(errorMessage);
				}
				this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.MaintenanceResults] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 255)));
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5684"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>PendingMaintenanceIssues property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5685"> 
		///		<ExpectedInput>Invalid string (256 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Pending Maintenance Issues
		/// </summary>
		public string PendingMaintenanceIssues
		{
			get
			{
				if (this._equipmentMaintenanceRow.IsNull(EQUIPMENTMAINTENANCE.PendingIssues))
					return string.Empty;

				return this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.PendingIssues].ToString();
			}
			set
			{
				if (value.Length > 255)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "255").ResString;
					throw new ArgumentException(errorMessage);
				}

				this._equipmentMaintenanceRow[EQUIPMENTMAINTENANCE.PendingIssues] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 255)));
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5686"> 
		///		<ExpectedInput>Empty data row</ExpectedInput>
		///		<ExpectedOutput>Populated equipment maintenance data row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5687"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads a data row with the data from the object
		/// </summary>
		/// <param name="dataRow">Equipment maintenance data row schema</param>
		/// <returns>A populated equipment maintenance data row</returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dataRow)
		{
			if (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.EquipmentMaintenanceGuid))
			{
				dataRow[EQUIPMENTMAINTENANCE.EquipmentMaintenanceGuid] = this.EquipmentMaintenceGuid;
			}

			if (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.EquipmentGuid))
			{
				dataRow[EQUIPMENTMAINTENANCE.EquipmentGuid] = this.EquipmentGuid;
			}

			if (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.MaintenanceDate))
			{
				dataRow[EQUIPMENTMAINTENANCE.MaintenanceDate] = this.MaintenanceDate;
			}

			if (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.MaintenanceResults))
			{
				dataRow[EQUIPMENTMAINTENANCE.MaintenanceResults] = this.MaintenanceResult;
			}

			if (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.MaintenanceType))
			{
				dataRow[EQUIPMENTMAINTENANCE.MaintenanceType] = this.MaintenanceActivity;
			}

			if (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.PendingIssues))
			{
				dataRow[EQUIPMENTMAINTENANCE.PendingIssues] = this.PendingMaintenanceIssues;
			}

			if (dataRow.Table.Columns.Contains(EQUIPMENT.DivisionCode))
			{
				dataRow[EQUIPMENT.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			}

			if (!this.IsNew && (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.RowVersion)))
			{
				dataRow[EQUIPMENTMAINTENANCE.RowVersion] = this.RowVersion;
			}

			return dataRow;
		}

		/// <summary>
		///  Loads the local object from a data row parameter
		/// </summary>
		/// <param name="dataRow">Equipment maintenance data row</param>
		protected override void LoadFromDataRow(System.Data.DataRow dataRow)
		{
			this._equipmentMaintenanceRow = dataRow;
			//
			if (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.MaintenanceDate))
			{
				if (!dataRow.IsNull(EQUIPMENTMAINTENANCE.MaintenanceDate))
                    this.MaintenanceDate = (System.DateTime) dataRow[EQUIPMENTMAINTENANCE.MaintenanceDate];
			}

			if (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.MaintenanceResults))
			{
				if (!dataRow.IsNull(EQUIPMENTMAINTENANCE.MaintenanceResults))
					this.MaintenanceResult = dataRow[EQUIPMENTMAINTENANCE.MaintenanceResults].ToString();
			}

			if (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.MaintenanceType))
			{
				if (!dataRow.IsNull(EQUIPMENTMAINTENANCE.MaintenanceType))
					this.MaintenanceActivity = dataRow[EQUIPMENTMAINTENANCE.MaintenanceType].ToString();
			}

			if (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.PendingIssues))
			{
				if (!dataRow.IsNull(EQUIPMENTMAINTENANCE.PendingIssues))
					this.PendingMaintenanceIssues = dataRow[EQUIPMENTMAINTENANCE.PendingIssues].ToString();
			}

			if (dataRow.Table.Columns.Contains(EQUIPMENTMAINTENANCE.RowVersion))
			{
				if (!dataRow.IsNull(EQUIPMENTMAINTENANCE.RowVersion))
				{
					this.RowVersion = (byte[])dataRow[EQUIPMENTMAINTENANCE.RowVersion];
				}
				//
				this.IsNew = dataRow.IsNull(EQUIPMENTMAINTENANCE.RowVersion);
			}
			else
			{
				this.IsNew = true;
			}
		}



		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5688"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Equipment maintenance data table for division</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5689"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty equipment maintenance data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function gets all equipment maintenance entries for a division
		/// </summary>
		/// <returns>Equipment maintenance data table</returns>
		public static DataTable GetEquipmentMaintenance()
		{
			DataTable dt = DAL.EquipmentMaintenance.GetEquipmentMaintenance();
			dt.AcceptChanges();
			return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5690"> 
		///		<ExpectedInput>Valid equipment guid</ExpectedInput>
		///		<ExpectedOutput>Equipment maintenance data table for equipment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5691"> 
		///		<ExpectedInput>Empty guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function gets all equipment maintenance entries for an equipment item
		/// </summary>
		/// <param name="equipmentGuid">Equipment unique identifier</param>
		/// <returns>Equipment maintenance data table</returns>
		public static DataTable GetEquipmentMaintenance(Guid equipmentGuid)
		{
			if (equipmentGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("equipment guid").ResString);
			}
			return DAL.EquipmentMaintenance.GetEquipmentMaintenance(equipmentGuid);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5692"> 
		///		<ExpectedInput>Valid equipment maintenance data table</ExpectedInput>
		///		<ExpectedOutput>Success indicator = true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5693"> 
		///		<ExpectedInput>Invalid equipment maintenance data table (duplicate entry included)</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves changes to an equipment maintenance data table
		/// </summary>
		/// <param name="dtEquipmentMaintenance">Equipment maintenance data table</param>
		/// <param name="lastUpdateFunctionId">Last update function</param>
		/// <returns>Success indicator</returns>
		public static bool Save(DataTable dtEquipmentMaintenance, Common.UpdateFunction lastUpdateFunctionId)
		{
			bool success = DAL.EquipmentMaintenance.Save(dtEquipmentMaintenance.GetChanges(DataRowState.Added),dtEquipmentMaintenance.GetChanges(DataRowState.Modified),lastUpdateFunctionId);
			if (success)
			{
				dtEquipmentMaintenance.AcceptChanges();
			}
			return success;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5694"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Maintenance type array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5695"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty array list</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function gets a list of previously entered maintenance types
		/// </summary>
		/// <returns>Array list of maintenance type strings</returns>
		public static ArrayList GetMaintenanceTypes()
		{
			DataTable dt = DAL.EquipmentMaintenance.GetMaintenanceTypes();
			//
			ArrayList al = new ArrayList(dt.Rows.Count);
			//
			foreach(DataRow dr in dt.Rows)
			{
				al.Add(dr[Common.VbecsTables.EquipmentMaintenance.MaintenanceType].ToString());
			}
			//
			return al;
		}
	}
}
